/* ***************************************************************** 
   MESQUITE -- The Mesh Quality Improvement Toolkit

   Copyright 2004 Sandia Corporation and Argonne National
   Laboratory.  Under the terms of Contract DE-AC04-94AL85000 
   with Sandia Corporation, the U.S. Government retains certain 
   rights in this software.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public License 
   (lgpl.txt) along with this library; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 
   diachin2@llnl.gov, djmelan@sandia.gov, mbrewer@sandia.gov, 
   pknupp@sandia.gov, tleurent@mcs.anl.gov, tmunson@mcs.anl.gov,
   kraftche@cae.wisc.edu         
   
   ***************************************************************** */
/*!
  \file   MeshBoundaryDomain2D.cpp
  \brief  


  \author Brian Miller
  \date   2010-10-10
*/
#ifdef PROFILING_ON
#include "TAU.h"
#endif

#include "Mesquite.hpp"
#include "Vector3D.hpp"
#include "MsqError.hpp"
#include "MsqVertex.hpp"
#include "MsqMatrix.hpp"
#include "MeshBoundaryDomain2D.hpp"
#include "PatchData.hpp"
#include "EdgeIterator.hpp"

#include <map>
#include <list>
#include <set>
#include <iostream>
using std::cout;
using std::endl;

#ifdef HAVE_IEEEFP_H
#  include <ieeefp.h>
#endif

#include <algorithm>

namespace MESQUITE_NS
{

#define DEBUG 1

// list of connected vertex handles and respective orientation indicator
typedef std::map<Mesh::VertexHandle,int > HandleSetMap;  


static void insert_edge(Mesh* mesh, TagHandle handle_set_map_tag, Mesh::VertexHandle v0, Mesh::VertexHandle v1)
{
    
  bool orient;
  Mesh::VertexHandle vs, vl;
  // find the smallest handle
  if (v0 < v1)
  {
    vs = v0;
    vl = v1;
    orient = true;
  }
  else
  {
    vs = v1;
    vl = v0;
    orient = false;
  }

  HandleSetMap* handle_set_map = 0;

  MsqError err;
  mesh->tag_get_vertex_data(handle_set_map_tag, 1, &vs, &handle_set_map, err);
  if (handle_set_map)
  {
        
    HandleSetMap::iterator connected_element = handle_set_map->find(vl);
    if (connected_element == handle_set_map->end())
    {            
      handle_set_map->insert(HandleSetMap::value_type(vl,(orient?1024:0)) );
    }
    else
    {           
      if (orient == !!(connected_element->second)) cout << "WARNING ... orientations are identical: " << vs << " " << vl << endl;

      handle_set_map->erase(connected_element);
    }
  }
  else
  {
    handle_set_map = new HandleSetMap();

    handle_set_map->insert(HandleSetMap::value_type(vl,(orient?1024:0)) );
    mesh->tag_set_vertex_data(handle_set_map_tag, 1, &vs, &handle_set_map, err);
  }
}


static void insert_edge(Mesh* mesh, TagHandle handle_set_map_tag, Mesh::VertexHandle v0, Mesh::VertexHandle v1, int material)
{
  if(0) cout<<"insert_edge ("<<v0<<","<<v1<<","<<material<<").  ";
    
  bool orient;
  Mesh::VertexHandle vs, vl;
  // find the smallest handle
  if (v0 < v1)
  {
    vs = v0;
    vl = v1;
    orient = true;
  }
  else
  {
    vs = v1;
    vl = v0;
    orient = false;
  }
  int oriented_material = material | (orient ? 1024 : 0);
  if(0) cout<<"edge:("<<vs<<","<<vl<<"), material:"<<material<<"  oriented_material: "<<oriented_material<<endl;
    
  HandleSetMap* handle_set_map = 0;

  MsqError err;
  mesh->tag_get_vertex_data(handle_set_map_tag, 1, &vs, &handle_set_map, err);
  if (handle_set_map)
  {

    HandleSetMap::iterator connected_element = handle_set_map->find(vl);
    if (connected_element == handle_set_map->end())
    {
      if(0) cout<<"   no data for edge ("<<vs<<";("<<vl<<","<<oriented_material<<") ), inserting new data into list."<<endl;
      handle_set_map->insert( HandleSetMap::value_type(vl,oriented_material) );
    }
    else
    {
      int other_oriented_material = connected_element->second;
      bool other_orient = ((other_oriented_material & 1024) == 1024);
      int other_material = other_oriented_material & 1023;
            
      if (orient == other_orient) cout << "WARNING ... orientations are identical: " << vs << " " << vl << endl;
            
      if (material == other_material)
      {
	if(0) cout<<"  found a duplicate edge("<<vs<<";("<<vl<<","<<oriented_material<<") ), removing it."<<endl;
	// an interior face ... just remove it
	handle_set_map->erase(connected_element);
      }
      else
      {
	//
	// found a pre-existing duplicate edge in another material.  We're going to choose to keep
	// only one of these edges and it will be the one with the lowest material ID
	//

	if( material < other_material ) 
	{
	  connected_element->second = oriented_material;
	}
                
	//
	// debugging start
	//
	if(0) 
	{
	  MsqVertex vert1,vert2;
	  mesh->vertices_get_coordinates(&vs, &vert1, 1, err);
	  if (err) {std::cout << err << std::endl; return; }
		    
	  mesh->vertices_get_coordinates(&vl, &vert2, 1, err);
	  if (err) {std::cout << err << std::endl; return; }
		    
	  cout<<"  found a material edge[ "<<vs<<":("<<vert1.x()<<","<<vert1.y()<<") - "<<vl<<":("<<vert2.x()<<","<<vert2.y()<<"), "<<oriented_material<<" ], keeping it."<<endl;
	}
	//
	// debugging end
	//

      }
    }
  }
  else
  {
    if(0) cout<<"   no handle map for "<<vs<<"  and therefore no data for edge ("<<vs<<";("<<vl<<","<<oriented_material<<") ), inserting new data into new list."<<endl;
    handle_set_map = new HandleSetMap();
    handle_set_map->insert( HandleSetMap::value_type(vl,oriented_material) );
    mesh->tag_set_vertex_data(handle_set_map_tag, 1, &vs, &handle_set_map, err);
  }

}


MeshBoundaryDomain2D::MeshBoundaryDomain2D(Plane orient, double offset, bool project_gradient, InterfaceSnapType theSnapType)
{
  mesh = 0;
  mesh_boundary_vertex_tag = 0;
  this->project_gradient = project_gradient;

  //
  // define the data needed to snap to a planar surface
  //
  mNormal = Vector3D(0.0,0.0,0.0);
  mNormal[orient] = 1.0;

  Vector3D point(0.0,0.0,0.0);
  point[orient] = offset;
    
  mCoeff = -(mNormal % point);

  //
  // initialization of the interface snapping parameters
  //
  mInterfaceSnapType = theSnapType;
  PICO=1.0e-12;
  MEGA=1.0e6;
  MICRO=1.0e-9;
  ITER_MAX=100;
}

MeshBoundaryDomain2D::~MeshBoundaryDomain2D()
{

  if (mesh && mesh_boundary_vertex_tag)
  {
    MsqError err;

    //
    // delete MyVertex pointers here before tag gets deleted
    //
    std::vector<MyVertex*> boundaryMyVertices(boundary_vertices.size());
    mesh->tag_get_vertex_data(mesh_boundary_vertex_tag, boundary_vertices.size(), &(boundary_vertices[0]),&(boundaryMyVertices[0]), err);
        
    for( size_t i=0;i<boundaryMyVertices.size(); i++)
    {
      if( (boundaryMyVertices[i]) != NULL )
      {
	delete boundaryMyVertices[i];
	boundaryMyVertices[i] = NULL;
      }   
    }
        
    mesh->tag_delete(mesh_boundary_vertex_tag, err);
    if (err) {std::cout << err << std::endl; return; }
  }
    
}


void MeshBoundaryDomain2D::skin_area_mesh(Mesh* mesh, float cos_crease_angle, const char* material_tag_name)
{
#ifdef PROFILING_ON
  TAU_PROFILE("MeshBoundaryDomain2D::skin_area_mesh()","",TAU_USER);
#endif
  MsqError err;

  // remove previous data

  boundary_vertices.clear();
  boundary_edges.clear();
  boundary_mesh.clear();
  corner_vertices.clear();

  // get all elements
  
  std::vector<Mesh::ElementHandle> elements;
  mesh->get_all_elements(elements, err);
  if (err) {std::cout << err << std::endl; return; }
  size_t num_elems = elements.size();

  // get all element topologies

  std::vector<EntityTopology> element_topologies(num_elems);
  mesh->elements_get_topologies(arrptr(elements), arrptr(element_topologies), num_elems, err);
  if (err) {std::cout << err << std::endl; return; }

  // if we have material tags then get them

  int *material_tag_data = 0;
  
  if (material_tag_name)
  {
    TagHandle material_tag = mesh->tag_get(material_tag_name, err);
    // if tag doesn't exist print warning
    if (!material_tag)
    {
      cout << "WARNING ... material tag " << material_tag_name << " does not exist" << endl;
      err.clear();
    }
    else
    {
      material_tag_data = new int[num_elems];
      mesh->tag_get_element_data(material_tag, num_elems, &elements[0], material_tag_data, err );
      if (err) {std::cout << err << std::endl; return; }
    }
  }

  // create a tag that will be used to associate a hash-like data structure with each vertex

  TagHandle handle_set_map_tag = mesh->tag_create("HANDLE_SET_MAP_TAG", Mesh::HANDLE, 1, 0, err);
  if (err) {std::cout << err << std::endl; return; }

  // insert the edges of all area elements into a hash-like data structure 

  for (size_t i = 0; i < num_elems; i++)
  {
    // get element's vertices
    std::vector<size_t> vtx_offsets;
    std::vector<Mesh::VertexHandle> adj_vertices;
    mesh->elements_get_attached_vertices(&elements[i],1,adj_vertices,vtx_offsets,err);
    if (err) {std::cout << err << std::endl; return; }

    size_t theNumVerts = adj_vertices.size();

    // Get element properties
    const EntityTopology &type = element_topologies[i];
    unsigned dim = TopologyInfo::dimension( type );
    int sides = TopologyInfo::adjacent( type, dim-1 );
    bool midedge, midface, midvol;
    TopologyInfo::higher_order( type, theNumVerts, midedge, midface, midvol, err );
    MSQ_ERRRTN(err);
    const bool midside = (dim == 2 && midedge);

    if(0) std::cout<<"This topology has "<<TopologyInfo::corners(type)<<" corners, "<<TopologyInfo::edges(type)<<" edges and "<<TopologyInfo::faces(type)<<" faces."<<std::endl;
           
    // For each side of the element (each edge for surface elems)..
    for (int j = 0; j < sides; ++j) {
      // Get the vertices of the side as indices into the above 'verts' list.
      unsigned n; // number of vertices 
      const unsigned* conn = TopologyInfo::side_vertices( type, dim-1, j, n, err );
      MSQ_ERRRTN(err);
		
      // grab higher-order node in center of side if it exists
      if (midside) {
	unsigned idx = TopologyInfo::higher_order_from_side( type, theNumVerts, dim-1, j, err );
	MSQ_ERRRTN(err);
	if (material_tag_data)
	{
	  insert_edge(mesh, handle_set_map_tag, adj_vertices[conn[0]], adj_vertices[idx], material_tag_data[i]);
	  insert_edge(mesh, handle_set_map_tag, adj_vertices[idx], adj_vertices[conn[1]], material_tag_data[i]);
	}
	else
	{
	  insert_edge(mesh, handle_set_map_tag, adj_vertices[conn[0]], adj_vertices[idx]);
	  insert_edge(mesh, handle_set_map_tag, adj_vertices[idx], adj_vertices[conn[1]]);
	}
      } else {
	// no higher order node at center of edge
	if (material_tag_data)
	{
	  insert_edge(mesh, handle_set_map_tag, adj_vertices[conn[0]], adj_vertices[conn[1]], material_tag_data[i]);
	}
	else
	{
	  insert_edge(mesh, handle_set_map_tag, adj_vertices[conn[0]], adj_vertices[conn[1]]);
	}
      }
		
    } // for (j in sides)

  }//end of i in elements 

  // get all vertices

  std::vector<Mesh::VertexHandle> vertices;
  mesh->get_all_vertices(vertices, err);
  if (err) {std::cout << err << std::endl; return; }
  size_t num_verts = vertices.size();

  HandleSetMap* handle_set_map = 0;

  size_t num_boundary_elems = 0;
  size_t num_boundary_verts = 0;

  // all edges that are still in the hash-like structure are boundary edges (or material boundary edges)
 
  for (size_t i = 0; i < num_verts; i++)
  {
    Mesh::VertexHandle v0 = vertices[i];
    handle_set_map = 0;
    mesh->tag_get_vertex_data(handle_set_map_tag, 1, &vertices[i], &handle_set_map, err);

    if (handle_set_map)
    {
      HandleSetMap::iterator map_element = handle_set_map->begin();
      while (map_element != handle_set_map->end())
      {
	Mesh::VertexHandle v1 = (*map_element).first;
                
	if( (*map_element).second & 1024 )
	{
	  boundary_edges.push_back(v0);
	  boundary_edges.push_back(v1);
	}
	else
	{
	  boundary_edges.push_back(v1);
	  boundary_edges.push_back(v0);
	}
                    
	num_boundary_elems++;

	map_element++;
      }
      delete handle_set_map;
    }
  }

  // the hash-like data structure no longer exists, the tag can be deleted

  mesh->tag_delete(handle_set_map_tag, err);

  // create a tag that will be used to associate a MyVertex struct with each boundary vertex

  mesh_boundary_vertex_tag = mesh->tag_create("MESH_BOUNDARY_VERTEX", Mesh::HANDLE, 1, 0, err);
  if (MSQ_CHKERR(err)) {std::cout << err << std::endl;}

  // create the "mesh" that has a MyVertex struct for each original boundary vertex to specify the mesh boundary domain that we snap to

  boundary_mesh.resize(boundary_edges.size());
  edge_normals.resize(boundary_edges.size()/2);


  MyVertex* my_vertex;
  for (size_t i = 0; i < boundary_edges.size()/2; i++)
  {
    //
    // set up myVertex structure for each edge vertex
    for (size_t k = 0; k < 2; k++)
    {
      my_vertex = 0;
      mesh->tag_get_vertex_data(mesh_boundary_vertex_tag, 1, &boundary_edges[i*2+k], &my_vertex, err);
      if (my_vertex == 0)
      {
	my_vertex = new MyVertex;
	my_vertex->closest = my_vertex;
	mesh->vertices_get_coordinates(&boundary_edges[i*2+k], &(my_vertex->coordinate), 1, err);
	my_vertex->snapped_to_position = my_vertex->coordinate;
	my_vertex->snapped_onto_edge = i;
	my_vertex->current_edge = i;
	my_vertex->edges=0;

	mesh->tag_set_vertex_data(mesh_boundary_vertex_tag, 1, &boundary_edges[i*2+k], &my_vertex, err);
	//
	// keep a list of unique boundary vertices here
	//
	boundary_vertices.push_back(boundary_edges[i*2+k]);
	num_boundary_verts++;
      }
      boundary_mesh[i*2+k] = my_vertex;
    }
  }

  // compute edge normals 
  for (size_t i = 0; i < boundary_edges.size()/2; i++)
  {
    MsqVertex ab = (boundary_mesh[i*2+1])->coordinate - (boundary_mesh[i*2])->coordinate;
    edge_normals[i] = MsqVertex(ab.y(),-ab.x(),ab.z());
    edge_normals[i].normalize();
    for (size_t k = 0; k < 2; k++)
    {
      // // if this vertex still has less than two edges
      // if ((boundary_mesh[i*2+k])->edges < 2)
      // {
      // if the first normal has not been set before
      if ((boundary_mesh[i*2+k])->edges == 0)
      {
	// set it now
	(boundary_mesh[i*2+k])->normal = edge_normals[i];
	(boundary_mesh[i*2+k])->edges = 1;
      }
      else if ((boundary_mesh[i*2+k])->edges == 1)
      {
	// set other edge information now                            
	(boundary_mesh[i*2+k])->other_edge = i;
	(boundary_mesh[i*2+k])->other_normal = edge_normals[i];
	(boundary_mesh[i*2+k])->edges = 2;

	// check if we have a corner
	// dot product of unit normals of adjacent edges
	float value = (boundary_mesh[i*2+k])->normal % edge_normals[i]; 
	if (value < cos_crease_angle)
	{
	  // sufficiently small angle cosine => corner
	  corner_vertices.insert(boundary_edges[i*2+k]);
	}
      } 
      else
      {
	(boundary_mesh[i*2+k])->edges = 3;
	// triple (or more) point => corner
	corner_vertices.insert(boundary_edges[i*2+k]);
      }
      // }
    }
  }


  //======================================
  // DEBUGGING OUTPUT START
  //======================================

  // cout<<"BOUNDARY_MESH:"<<endl;
  // for(std::vector<MyVertex*>::const_iterator it= boundary_mesh.begin();
  //     it != boundary_mesh.end(); ++it)
  // {
  //   cout<<"coordinate:("<<(*it)->coordinate.x()<<","<<(*it)->coordinate.y()<<","<<(*it)->coordinate.z()<<")  ";
  //   cout<<" edges="<<(*it)->edges<<"  current_edge:"<<(*it)->current_edge<<"  normal:"<<(*it)->normal;
  //   if( (*it)->edges > 1 )
  //   {
  //     cout<<"  other_edge:"<<(*it)->other_edge<<"  other_normal: "<<(*it)->other_normal<<endl;
  //   }
  //   else
  //   {
  //     cout<<endl;
  //   }
        
        
  // }

  // size_t numCornerVertices = corner_vertices.size();
    
  // std::vector<MsqVertex> corner_vertex_coords(numCornerVertices);

  // cout<<"CORNER_VERTICES ("<<numCornerVertices<<" of them):"<<endl;
  // size_t i=0;
  // for(std::set<Mesh::VertexHandle>::const_iterator it=corner_vertices.begin();
  //     it != corner_vertices.end(); it++,i++)
  // {
  //   mesh->vertices_get_coordinates(&(*it), &corner_vertex_coords[i], 1, err);
  //   cout<<(*it)<<"  "<<corner_vertex_coords[i]<<endl;
  // }
  // cout<<endl;   
  //======================================
  // DEBUGGING OUTPUT END
  //======================================

  if( material_tag_data ) delete[] material_tag_data;

  this->mesh = mesh;
}

void MeshBoundaryDomain2D::copy_surface_mesh(Mesh* mesh)
{
  this->mesh = mesh;
}

static float distance(Vector3D& P, MsqVertex& A, MsqVertex& B, Vector3D& P_new) 
{
  // the line through A and B is parametrized as L(t) = B + tM
  Vector3D M = A - B;
  Vector3D T;

  double t0 = (M%(P-B)) / (M%M);
  if (t0 < 0) T = B;
  else if (t0 > 1) T = A;
  else T = B+M*t0;

  P_new = T;

  return (T-P)%(T-P);
}


void MeshBoundaryDomain2D::snap_to(Mesh::VertexHandle vertex_handle, Vector3D &coordinate) const
{
// #ifdef PROFILING_ON
//   TAU_PROFILE("MeshBoundaryDomain2D::snap_to()","",TAU_USER);
// #endif
    
  MsqError err;
  MyVertex* my_vertex = 0;

  // get the MyVertex record that
  //   (a) specifies where this vertex is currently attached to the boundary mesh
  //   (b) is itself part of the definition of the bounday mesh

  mesh->tag_get_vertex_data(mesh_boundary_vertex_tag, 1, &vertex_handle, &my_vertex, err);
  if (err && (err.error_code() != MsqError::TAG_NOT_FOUND) ) {std::cout << err << std::endl; return; }

  //
  // We alternatively snap to an interface if there is an associated MyVertex record for this vertex
  // meaning that it does live on a snapping edge.  If there is no MyVertex record then it is an interor
  // vertex so we project it onto the plane
  //

  if (my_vertex)
  {
    if( mInterfaceSnapType == LINEAR )
    {

      double d;
      double d_min = 100000000000000000.0f;
      int i_min = -1;
      MsqVertex min;
      MsqVertex snapped;

      // get the current position of the vertex in the *MsqMesh* (not the patch)
 
      MsqVertex current_position;
      mesh->vertices_get_coordinates(&vertex_handle, &current_position, 1, err);
      if (err) {std::cout << err << std::endl; return; }

      // depending on whether the vertex is in a smooth area or along a crease we process it differently


      if (my_vertex->edges == 2) // snap vertex in smooth area (not at a corner, those will be fixed)
      {
	// if the current position of the vertex equals the snapped position the snapped face becomes the current face 
	if (current_position == my_vertex->snapped_to_position)
	{
	  // this "hack" for updating the current edge becomes necessary as mesquite does
	  // not communicate back to the snap function whether the returned "snapped to position"
	  //  was used or whether it lead to inverted elements / worse quality and is therefore rejected
	  my_vertex->current_edge = my_vertex->snapped_onto_edge;
	}
	int edgeIndices[2]={my_vertex->current_edge, my_vertex->other_edge};
            
	for( int edge=0;edge<2;edge++)
	{
	  int i=edgeIndices[edge];
                
	  d = distance(coordinate, (boundary_mesh[2*i])->coordinate, (boundary_mesh[2*i+1])->coordinate, snapped);
	  if (d < d_min)
	  {
	    min = snapped;
	    d_min = d;
	    i_min = i;
	  }
	}

	// store the triangle we snapped onto
	my_vertex->snapped_onto_edge = i_min;
	// store the position we snapped to
	my_vertex->snapped_to_position = min;
      }

      coordinate = min;
    } 
    else if( mInterfaceSnapType == QUADRATIC )
    {

      MsqVertex snapped;

      MsqVertex position_1,position_0,position_2;

      mesh->vertices_get_coordinates(&vertex_handle, &position_1, 1, err);
      if (err) {std::cout << err << std::endl; return; }

      if (my_vertex->edges == 2) // snap vertex in smooth area (not at a corner, those will be fixed)
      {
	// if the current position of the vertex equals the snapped position the snapped face becomes the current face 
	if (position_1 == my_vertex->snapped_to_position)
	{
	  // this "hack" for updating the current edge becomes necessary as mesquite does
	  // not communicate back to the snap function whether the returned "snapped to position"
	  //  was used or whether it lead to inverted elements / worse quality and is therefore rejected
	  my_vertex->current_edge = my_vertex->snapped_onto_edge;
	}

	//
	// get the handles and positions of the two other vertices attached to vertex_handle
	//
	Mesh::VertexHandle handle_0, handle_2;
            
	if( boundary_edges[2*my_vertex->current_edge] == vertex_handle )
	{
	  handle_0 = boundary_edges[2*my_vertex->current_edge+1];
	}
	else
	{
	  handle_0 = boundary_edges[2*my_vertex->current_edge];
	}
	mesh->vertices_get_coordinates(&handle_0, &position_0, 1, err);
	if (err) {std::cout << err << std::endl; return; }

	if( boundary_edges[2*my_vertex->other_edge] == vertex_handle )
	{
	  handle_2 = boundary_edges[2*my_vertex->other_edge+1];
	}
	else
	{
	  handle_2 = boundary_edges[2*my_vertex->other_edge];
	}
	mesh->vertices_get_coordinates(&handle_2, &position_2, 1, err);
	if (err) {std::cout << err << std::endl; return; }

	double x[3]={position_0.x(),position_1.x(),position_2.x()};
	double y[3]={position_0.y(),position_1.y(),position_2.y()};
	double xs = coordinate.x();
	double ys = coordinate.y();
            
	quadraticSnap(&xs, &ys, &x[0], &y[0]);
	coordinate.x(xs);
	coordinate.y(ys);
      }
        
    } 
    else if( mInterfaceSnapType == MOMENTPRESERVING )
    {
      cout<<" UNIMPLEMENTED FOR NOW." <<endl;
    }
    else
    {
      cout<<" INVALID INTERFACE SNAPPING TYPE CHOSEN!"<<endl;
    }
  }
  
  // 
  // In any case, project back to planar domain
  // 
  coordinate -= mNormal * ( mNormal % coordinate + mCoeff );

}

void MeshBoundaryDomain2D::vertex_normal_at(Mesh::VertexHandle vertex_handle,
					    Vector3D &coordinate) const
{
  coordinate = mNormal;
}

void MeshBoundaryDomain2D::element_normal_at(Mesh::ElementHandle h,
					     Vector3D &coordinate) const
{
  coordinate = mNormal;
}

void MeshBoundaryDomain2D::vertex_normal_at( 
					    const Mesh::VertexHandle* handles,
					    Vector3D coords[],
					    unsigned count,
					    MsqError& ) const
{

  for (unsigned i = 0; i < count; ++i)
    coords[i] = mNormal;
}

void MeshBoundaryDomain2D::closest_point( Mesh::VertexHandle handle,
					  const Vector3D& position,
					  Vector3D& closest,
					  Vector3D& normal,
					  MsqError& ) const
{
  normal = mNormal;
  snap_to(handle, closest);
  // closest = position - mNormal * (mNormal % position + mCoeff);
}


void MeshBoundaryDomain2D::domain_DoF( const Mesh::VertexHandle* handles,
				       unsigned short* dof_array,
				       size_t num_vertices,
				       MsqError&  ) const
{

  std::fill( dof_array, dof_array + num_vertices, 2 );
  // if (project_gradient)
  //   std::fill( dof_array, dof_array + num_vertices, 99 );
  // else
  //   std::fill( dof_array, dof_array + num_vertices, 1 );

}
 
void 
MeshBoundaryDomain2D::conservedFitting(double *xs, double *ys, double x[4], double y[4])
{
#ifdef PROFILING_ON
  TAU_PROFILE("MeshBoundaryDomain2D::conservedFitting()","",TAU_USER);
#endif

  // first fit the given linear facets with an LS method.
  // to do this, set a normal (by fitting a straight line
  // to the points, or simply taking the nomal of the middle
  // facet, now try the simplest).
   
  double QUARTER = 0.25;
  double HALF = 0.5;
  double ONETHIRD = 1.0/3.0;

  double dx = x[3] - x[0];
  double dy = y[3] - y[0];
  double length = sqrt(dx*dx + dy*dy);

  // (n, t) is a right hand frame (surface normal frame). 
  double tx = dx/length;
  double ty = dy/length;

  double nx = -ty;
  double ny = +tx;

  // now normal is defined, map points into normal frame (origin at
  // (xs, ys), say. (coordinate transform from global to local). 

  double xp[4], yp[4];

  for(int i = 0; i < 4; i++) {

    dx = x[i] - (*xs);
    dy = y[i] - (*ys);

    xp[i] = dx*tx + dy*ty;
    yp[i] = dx*nx + dy*ny;
  }

  // ready to do LS length fitting:
  // the domain of integration is three intervals (xp[0], xp[1]),
  // (xp[1], xp[2]), and (xp[2], xp[3]), the integral function
  // on each interval would be:  (Q - N_k)^2/2, where Q is quadratic
  // that Q = a t^2 + b t + c, N is linear: N = alpha_k t + beta_k.
  // The base integrals are integrals of: (t^4, t^3, t^2, t, 1).
   
  // limits of domain of integrations. 

  double xL, xR;
  double xL2, xR2;
  double xL3, xR3;
  double xL4, xR4;

  xL = xp[0];
  xR = xp[3]; 

  xL2 = xL*xL;
  xR2 = xR*xR;
 
  xL3 = xL2*xL;
  xR3 = xR2*xR;
 
  xL4 = xL2*xL2;
  xR4 = xR2*xR2;

  // the matrix for LS fitting is easy to obtain, which is:
  //                             
  //           | D^5/5, D^4/4, D^3/3 |
  //       M = | D^4/4, D^3/3, D^2/2 |
  //           | D^3/3, D^2/2, D^1/1 |
  //                               
   

  double **M = (double **) malloc(3*sizeof(double *));
 
  for(int i = 0; i < 3; i++) {
   
    M[i] = (double *) malloc(3*sizeof(double));
  }
 
  M[0][0] = 0.2*(xR3*xR2 - xL3*xL2);
  M[0][1] = QUARTER*(xR4 - xL4);
  M[0][2] = ONETHIRD*(xR3 - xL3);

  M[1][0] = M[0][1];
  M[1][1] = M[0][2];
  M[1][2] = HALF*(xR2 - xL2);

  M[2][0] = M[0][2];
  M[2][1] = M[1][2];
  M[2][2] = xR - xL;

  // the right hand side entries are:
  //
  //       R_1 = Sum{alpha[i] (D_i)^4/4 + beta[i] (D_i)^3/3:
  //                 i = 1 through 3}
  //       R_2 = Sum{alpha[i] (D_i)^3/3 + beta[i] (D_i)^2/2:
  //                 i = 1 through 3}
  //       R_3 = Sum{alpha[i] (D_i)^2/2 + beta[i] (D_i)^1/1:
  //                 i = 1 through 3}
   

  // I still need to calculate alpha_i, beta_i. 

  // parameters to define facets in normal frame. 
 
  double alfa[3];
  double beta[3];

  for(int i = 0; i < 3; i++) {

    dx = xp[i + 1] - xp[i];
    dy = yp[i + 1] - yp[i];
   
    alfa[i] = dy/dx;
    beta[i] = yp[i] - xp[i]*alfa[i];
  }

  // R: the right hand side. 
  double *R = (double *) malloc(3*sizeof(double));

  // initialize. 
  R[0] = 0.0; R[1] = 0.0; R[2] = 0.0;

  for(int i = 0; i < 3; i++) {

    xL = xp[i];
    xR = xp[i + 1];
   
    xL2 = xL*xL;
    xR2 = xR*xR;

    xL3 = xL2*xL;
    xR3 = xR2*xR;

    xL4 = xL2*xL2;
    xR4 = xR2*xR2;

    R[0] += QUARTER*alfa[i]*(xR4 - xL4) + ONETHIRD*beta[i]*(xR3 - xL3);
    R[1] += ONETHIRD*alfa[i]*(xR3 - xL3) + HALF*beta[i]*(xR2 - xL2);
    R[2] += HALF*alfa[i]*(xR2 - xL2) + beta[i]*(xR - xL);
  }

  // solve for {a, b, c}. 
  CholeskyInverse(M, R, 3);

  double a = R[0];
  double b = R[1];
  double c = R[2];

  // the volume/moment conserved quadratic surface is:
  // n = a t^2 + b t + c. I project (xs, ys) onto it.
  // be aware that (xs, ys) is mapped to (0, 0) already.
   

  // the projection is obtained by solving the cubic:
  // f(t) = (2 a t + b) (a t^2 + b t + c) + t = 0.
   

  // initial guess (the origin = the second order solution).
   

  double t = 0.0;
  double n = 0.0;
  double diff = MEGA;
   
  int iter = 0;

  while (diff > MICRO && iter < ITER_MAX) {
    n = a*t*t + b*t + c;
    double nt = 2*a*t + b;
    double f = n*nt + t;
    double df = nt*nt + 2*a*n + 1.0;
    // double ddf = 6*a*nt;

    // Newton's Method (second order). 
    t -= f/df;
    iter++;

    // Jin's method (fifth order). (later). 
  }
 
  // check solution. 

  length = sqrt(n*n + t*t);

  // tengent. 
  length = sqrt(pow(2*a*t + b, 2) + 1.0);

  // convert the solution (n, t) back to physical space. 

  *xs += t*ny + n*nx;
  *ys += t*ty + n*tx;

  // check the solution next. 

  // is volume/moment conserved? 

  double vL = 0.0; // volume under linear facets. 
  double mL1 = 0.0; // first momentum under linear facets. 
  double mL2 = 0.0; // second momentum under linear facets. 

  // compute vL. 

  for(int i = 0; i < 3; i++) {
   
    xL = xp[i];
    xR = xp[i + 1];

    xL2 = xL*xL;
    xR2 = xR*xR;

    xL3 = xL2*xL;
    xR3 = xR2*xR;

    xL4 = xL2*xL2;
    xR4 = xR2*xR2;

    vL += HALF*(xR*xR - xL*xL)*alfa[i] + (xR - xL)*beta[i];
    mL1 += ONETHIRD*(xR3 - xL3)*alfa[i] + HALF*(xR2 - xL2)*beta[i];
    mL2 += QUARTER*(xR4 - xL4)*alfa[i] + ONETHIRD*(xR3 - xL3)*beta[i];
  }

  // then vQ. 
  double vQ = 0.0; // volume under quadratic curve. 
  double mQ1 = 0.0; // first momentum under quadratic curve. 
  double mQ2 = 0.0; // second momentum under quadratic curve. 

  xL = xp[0];
  xR = xp[3];

  xL2 = xL*xL;
  xR2 = xR*xR;

  xL3 = xL2*xL;
  xR3 = xR2*xR;
 
  xL4 = xL2*xL2;
  xR4 = xR2*xR2;

  double xL5 = xL2*xL3;
  double xR5 = xR2*xR3;

  vQ  = a*ONETHIRD*(xR3 - xL3) + b*HALF*(xR2 - xL2) + c*(xR - xL);
  mQ1 = a*QUARTER*(xR4 - xL4) + b*ONETHIRD*(xR3 - xL3) + HALF*c*(xR2 - xL2);
  mQ2 = a*0.2*(xR5 - xL5) + b*QUARTER*(xR4 - xL4) + c*ONETHIRD*(xR3 - xL3);

 }


///***************************************************
///
/// @author Jin Yao
/// @param c The 3 coefficients of the equation and the two returned solutions.
/// @pre c != 0
/// @post none
///
void 
MeshBoundaryDomain2D::QuadraticSolver(double *c)
{
  // the equation to solve is c[0] + c[1]x + c[2] x^2 = 0. 
  // output roots c[0] and c[1]. 

  double scale = fabs(c[0]) + fabs(c[1]) + fabs(c[2]);

  if(scale < PICO)
  {
    cout<<"QuadraticSolver coefficients too small, not eligible!"<<endl;
    return ;
  }

  c[0] /= scale;
  c[1] /= scale;
  c[2] /= scale;

  double delta = c[1]*c[1] - 4*c[0]*c[2];

  // If delta < 0.0 there are no roots.
  // If delta == 0.0 there are double roots (degenerate quadradic).
  // Otherwise there are 2 unique roots.
  if (delta < 0.0) {
    c[0] = -MEGA;
    c[1] = -MEGA;
    return;
  }
  else if (delta == 0.0) {
    c[0] = -0.5*c[1]/c[2];
    c[1] = c[0];
    return;
  }
  else {
    // If c[2] is too small consider it to be 0.0 so the equation is linear
    // and there is only one root, c[0].  Set c[1] to -MEGA in this case. 
    if (fabs(c[2]) > PICO) {
      c[0] = 0.5*(-c[1] + sqrt(delta))/c[2];
      c[1] = -c[1]/c[2] - c[0];
    }
    else {
      c[0] = -c[0]/c[1];
      c[1] = -MEGA;
    }
  }

  return;
}



//****************************************************
///
/// @author Jin Yao
/// @param a rank x rank matrix "a" in a.lambda = lambda
/// @param b "lambda" in a.lambda = lambda
/// @param rank Rank of equation being solved.
/// @pre rank != 0
/// @pre a != 0
/// @pre b != 0
/// @post none
///
void 
MeshBoundaryDomain2D::CholeskyInverse(double **a, double *b, int rank)
{
  int i, j, k;

  for (j = 0; j < rank; ++j) {   
    for (k = 0; k < j; ++k) {
      for (i = j; i < rank; ++i) {
	a[i][j] = a[i][j] - a[i][k]*a[j][k];
      }
    }

    // SVD treatment. 
    if (a[j][j] < PICO)
      a[j][j] = MEGA*MEGA*MEGA;

    a[j][j] = sqrt(a[j][j]);       

    for (k = j + 1; k < rank; ++k) {
      a[k][j] = a[k][j]/a[j][j];
    }
  }

  // Now a[i][j] are the lower triangle matrix. 

  // so I am going to solve L.b = lambda and then L^T.lambda = b. 

  // step1: 
  for (i = 0; i < rank; ++i) {
    if (i > 0) {
      for (j = 0; j < i; ++j) {
	b[i] -= a[i][j]*b[j];
      }
    }

    b[i] /= a[i][i];
  }

  //  step2: 
  for (i = rank - 1; i >= 0; --i) {
    if (i < rank - 1) {
      for (j = rank - 1; j > i; --j) {
	b[i] = b[i] - a[j][i]*b[j];
      }
    }
    b[i] = b[i]/a[i][i];
  }

  // Now "b" is the solution of this least-squared problem. 
  return;
}

void MeshBoundaryDomain2D::quadraticSnap(double *xs, double *ys, double x[3], double y[3]) const
{
// #ifdef PROFILING_ON
//   TAU_PROFILE("MeshBoundaryDomain2D::quadraticSnap()","",TAU_USER);
// #endif

  //
  // projecting (xs,ys) onto quadratic passing through (x[0],y[0]) (x[1],y[1]) and (x[2],y[2])
  //

  // 
  // first define angle of rotation from positive x-axis to secant segment which  
  // along with (x[1],y[1]) will define our local coordinate system.
  //
  double theta = -atan2(y[2]-y[0],x[2]-x[0]); // angle between (x[0],y[0]), (x[2],y[2]) and x-axis
  MsqMatrix<2,2> rot;
  rot(0,0)=cos(theta); rot(0,1)=-sin(theta);
  rot(1,0)=sin(theta); rot(1,1)=cos(theta);
    
  //
  // w0=(eta(0),xi(0)) and w2=(eta(2),xi(2) are 
  // (x[0],y[0]) and (x[2],y[2]) tranformed into 
  // local coordinates.  (w1=(0,0) in local coordinates)
  //
  MsqVector<2> w0,w2;
  w0(0) = rot(0,0)*(x[0]-x[1]) + rot(0,1)*(y[0]-y[1]); 
  w0(1) = rot(1,0)*(x[0]-x[1]) + rot(1,1)*(y[0]-y[1]);

  w2(0) = rot(0,0)*(x[2]-x[1]) + rot(0,1)*(y[2]-y[1]);
  w2(1) = rot(1,0)*(x[2]-x[1]) + rot(1,1)*(y[2]-y[1]);

  //
  // the quadratic in local coordinates (xi,eta) is 
  // q(eta) = 0 + q1*eta * q2 * eta^2 where (q1,q2) satisify
  //
  // [ xi(0) ]  [ eta(0)  eta(0)^2 ][q1]
  // [ xi(2) ] =[ eta(2)  eta(2)^2 ][q2]
  //
  double invDetQ = 1.0/(w0(0)*w2(0)*w2(0) - w2(0)*w0(0)*w0(0));
  double q1 = invDetQ * (w0(1)*w2(0)*w2(0) - w2(1) * w0(0)*w0(0) );
  double q2 = invDetQ * (-w0(1)*w2(0) +w2(1)*w0(0));
    
  //
  // ws is the point to snap transformed to local coordinates
  MsqVector<2> ws;
  ws(0) = rot(0,0)*(*xs-x[1]) + rot(0,1)*(*ys-y[1]);
  ws(1) = rot(1,0)*(*xs-x[1]) + rot(1,1)*(*ys-y[1]);
    

  //
  // The distance from ws to the quadratic is found 
  // by minimizing the distance squared given by:
  //
  // 0 = a0 + a1*eta + a2*eta^2 + a3*eta^3
  //
  double a0 = -2.0*(ws(0)+ws(1)*q1);
  double a1 = 2.0+2.0*q1*q1-4.0*ws(1)*q2;
  double a2 = 6.0*q1*q2;
  double a3 = 4.0*q2*q2;
    
  //
  // (eta,xi) will the computed coordinates of ws snapped to the
  // quadratic in local coordinates.  
  // eta and dEta compute the Newton iterates initialized to 0 since
  // we're using Newton's method to solve the min distance problem.
  //
  double eta=0.0, dEta=1.0e12, xi=0.0;
    
  while( fabs(dEta) > .0001*fabs(w2(0)-w0(0)) )
  {
    dEta = (a0+eta*(a1+eta*(a2+a3*eta)))/(a1+eta*(2.0*a2+3.0*a3*eta));
    eta -= dEta;
  }
    
  xi = eta*(q1+q2*eta);
    
  *xs = rot(0,0)*eta + rot(1,0)*xi + x[1];
  *ys = rot(0,1)*eta + rot(1,1)*xi + y[1];
    
    
}


} // namespace Mesquite
